/* Copyright (C) 2011-2018 RealVNC Ltd.  All Rights Reserved.
 */

#ifndef __VNCDISCOVERYSDK_H__
#define __VNCDISCOVERYSDK_H__

#include <vncdiscoverysdktypes.h>
#include <vncdiscoverysdkcallbacks.h>
#include <string.h>

#include <vncdiscoverer.h>
#include <vncdiscoverysdkutils.h>

/**
* \file vncdiscoverysdk.h
*
* This is the main header file for the VNC Discovery SDK.  To use the SDK,
* you need only include this file.
*
* This file defines the VNC Discovery SDK Interface, the data structure
* associated with it and the sole entry point for the Discovery SDK DLL or
* shared object.
*
* \see VNCDiscoverySDKInitialize, VNCDiscoverySDK
* \see vncdiscoverysdkcallbacks.h, vncdiscoverysdktypes.h
*/

/**
 * \mainpage VNC Discovery SDK Native Interface
 *
 * \section section_contents Contents
 *
 *  -# \ref section_overview
 *  -# \ref section_usage
 *  -# \ref section_memory
 *  -# \ref section_thread
 *  -# \ref section_legal
 *
 * \section section_overview Overview
 *
 * The Discovery SDK is for use within viewer applications and allows
 * discovering entities. An entity represents a dynamically discovered
 * component or device. It could be a MirrorLink server, a generic USB device,
 * or some other piece of hardware.
 *
 * The SDK abstracts away from the medium with which the connection to the
 * entity is made. It allows plug-ins to be loaded dynamically or statically
 * linked against. A plug-in is called a Discoverer and the user of the SDK -
 * the client application - can load a specific plug-in by specifying its type.
 * It is the plug-ins that do the actual discovery of entities. Consult the
 * documentation for each discoverer to understand the different entities that
 * it can report. For documentation regarding the development of Discoverer
 * plug-ins please see the "VNC Discovery SDK - Native Discoverer Plug-in
 * Interface" document. This document covers only the client API interface of
 * the VNC Discovery SDK.
 *
 * For entities that contain a VNC server, the Discovery SDK will provide a
 * vnccmd string which can be passed to the VNC Viewer SDK to establish a VNC
 * connection to the server. The VNC Discovery SDK can run in parallel with the
 * VNC Viewer SDK and discovering of new entities can happen while the Viewer
 * is connected to an existing server. Several VNC connections can be run in
 * parallel by using several instances of the VNC Viewer SDK. Please consult
 * the VNC Viewer SDK for more information.
 * 
 * The native interface to the SDK does not provide any user interface of its
 * own.
 *
 * \section section_usage Basic usage
 *
 * \subsection subsection_usage_compiling Compiling your application
 *
 * To use the SDK, your application must first define exactly one of the
 * preprocessor macros VNC_USE_STDINT_H or VNC_USE_DEFAULT_INT_TYPEDEFS.  These
 * macros govern the definitions of the vnc_*_t integer types defined by
 * vncint.h.
 *
 *  - If your build environment provides the ISO C99 header stdint.h, or you
 *    can author an equivalent yourself, you should define VNC_USE_STDINT_H.
 *    This is the recommended approach for defining the SDK's integer types.
 *  - If you examine vncint.h and find that the types used when you define
 *    VNC_USE_DEFAULT_INT_TYPEDEFS are correct for your platform, then you may
 *    wish to define VNC_USE_DEFAULT_INT_TYPEDEFS instead of VNC_USE_STDINT_H.
 *    Additionally, if you are targeting a 64-bit platform with a compiler
 *    other than Microsoft Visual C++ or gcc, then you may have to define
 *    VNC_FORCE_64_BIT in order to define vnc_intptr_t and vnc_uintptr_t
 *    correctly.  See vncint.h itself for further information.
 *
 * The types defined by VNC_USE_DEFAULT_INT_TYPEDEFS are known to be correct
 * for Microsoft Visual C++ 2005.
 *
 * After defining either VNC_USE_STDINT_H or VNC_USE_DEFAULT_INT_TYPEDEFS, your
 * application should include vncdiscoverysdk.h.  You are encouraged to use the
 * VNCVerifyIntegerTypes() macro to verify that the integer types have been
 * defined correctly.
 *
 * \subsection subsection_usage_startup Initializing the SDK
 *
 * -# Load the SDK DLL (necessary only if you did not link against it).
 *    - on Windows, call LoadLibrary()
 *    - on Linux, call dlopen()
 * -# Locate the SDK entry point, VNCDiscoverySDKInitialize().
 *    - on Windows, call GetProcAddress()
 *    - on Linux, call dlsym()
 * -# Call VNCDiscoverySDKInitialize(). This populates a VNCDiscoverySDK
 *    structure with pointers to the rest of the SDK's 'methods'. Also make
 *    sure you pass to this method the structure containing the callback
 *    function pointers. The SDK will use this structure for communicating back
 *    to the client application. The application can choose to be called back
 *    only on some events; it doesn't need to listen to all of them. As a
 *    minimum it would make sense to have
 *    VNCDiscoverySDKEntityAppearedCallback().
 *
 * Typically, your application will call VNCDiscoverySDKInitialize() at
 * start-up.
 *
 * After the start-up, you should configure the Discovery SDK parameters,
 * through VNCDiscoverySDKSetParameter().
 *
 * \subsection subsection_usage_licensing Licensing the SDK
 *
 * In order to use the SDK, the application must always provide each SDK
 * instance with at least one license. Use VNCDiscoverySDKAddLicense to do
 * this. Contact your RealVNC sales representative to obtain a license.
 *
 * \subsection subsection_usage_creating Creating a VNC Discoverer object
 * 
 * If necessary, call VNCDiscoverySDKRegisterDiscoverer() to register a type of
 * Discoverer. This is only necessary if you have statically linked against the
 * Discoverer.
 *
 * Call VNCDiscoverySDKCreateDiscoverer() to create a new Discoverer. The
 * Discoverer could be tied to a specific transport medium, or a specific
 * protocol, or could be more generic. An application can create several
 * Discoverers.
 *
 * Once the Discoverer is created the client application should provide it, if
 * it wishes, with a context that will be passed back to the application in the
 * callbacks. This is done through VNCDiscoverySDKSetContext().
 *
 * Some Discoverers have various settings that could be configured. This would
 * be done through VNCDiscoverySDKDiscovererSetProperty() once the Discoverer 
 * is created. Also the client could see how the Discoverer is configured to
 * run, or what it supports, through calling
 * VNCDiscoverySDKDiscovererGetProperty().
 *
 * \subsection subsection_usage_starting_discovering Starting discovering
 *
 * The client asks the SDK to start a Discoverer through 
 * VNCDiscoverySDKDiscovererStart() or VNCDiscoverySDKDiscovererStartEx().
 * Whenever the Discoverer finds an entity it will notify the client through
 * the SDK (an Entity Appeared notification will be made through
 * VNCDiscoverySDKEntityAppearedCallback()).
 *
 * Any number of Discoverers can be discovering concurrently.
 *
 * \subsection subsection_usage_entities Working with Entities
 *
 * The client can use an Entity to obtain information about the discovered
 * component/device, such as an ID, a name, or - in the case of VNC servers - a
 * vnccmd string. This information can be retrieved synchronously, through
 * VNCDiscoverySDKFetchEntityValueBlocking(), or asynchronously, through
 * VNCDiscoverySDKFetchEntityValue() with the callback
 * VNCDiscoverySDKFetchEntityValueCallback(). The information is organised as
 * key-value pairs, where both the keys and the value are strings.
 * 
 * Usually an application will want to know the ID, type, and name of an
 * Entity. These can be retrieved by using the keys 'id', 'type', and 'name' in
 * the fetch requests. The 'id' and 'type' keys can be received straight away,
 * so the synchronous call will be quick and is recommended. For the 'name' key
 * the fetch can take some long time to resolve so it is recommended to use the
 * asynchronous version.
 *
 * There can be other key-value pairs available for entities. Read the
 * Discoverer-specific documentation for details of these.
 *
 * Some Discoverers support setting key-value information synchronously,
 * through VNCDiscoverySDKPostEntityValueBlocking(), or asynchronously, through
 * VNCDiscoverySDKPostEntityValue(). For example if a server should be renamed
 * the client can post to key 'name' value 'NewName'. It is recommended to use
 * the asynchronous post most of the time.
 *
 * The client application should work with an Entity while the Discoverer that
 * owns it is still running. If the Discoverer is stopped then values can't be
 * fetched from the Entity. The only requests available on an Entity when the
 * Discoverer is not running are the ones related to context data: 
 * VNCDiscoverySDKEntitySetContext() and
 * VNCDiscoverySDKEntityGetContext(). The context data can be associated by
 * the application with an Entity for convenience. 
 *
 * \subsection subsection_usage_vnc VNC Servers
 *
 * Use VNCDiscoverySDKFetchEntityValue() to fetch a vnccmd string from the
 * entity.
 *
 * Once the client application has the vnccmd it can pass it to the VNC Viewer
 * SDK to ask the Viewer SDK to connect to the server. See the VNC Viewer SDK
 * documentation for usage of the Viewer.
 *
 * \note The existence of an entity is not definite, even if a vnccmd could be
 * retrieved successfully. It is not guaranteed that a VNC Server is present
 * and listening for incoming connections. False positives might be reported by
 * some Discoverers.
 *
 * Sometimes the VNC Viewer SDK requires for establishing a connection, in
 * addition to 'vnccmd', a series of parameters. They can be retrieved from the
 * Discoverer through the VNCDiscoverySDKGetViewerParameters() call.
 *
 * \subsection subsection_usage_mirrorlink MirrorLink Entities
 *
 * If you are using a Discoverer that supports MirrorLink then its Entities can
 * also support setting string key-value pairs. This can be done through the
 * Post Entity Value calls. The asynchronous method is recommended.
 *
 * A client application would use the post method for example for starting an
 * application on the MirrorLink device.
 *
 * Most Discoverers will not have MirrorLink functionality. Refer to the
 * Discoverer specific documentation.
 *
 * \subsection subsection_usage_addressbook Address Book Entities
 *
 * If the client application is using an Address Book Discoverer in order to
 * store information about VNC Servers it connected to, then it can ask that
 * Discoverer to create entities, through VNCDiscoverySDKCreateEntity(). Once
 * the Entity is created, the application can write to it by using one of the
 * Post methods (the synchronous version is recommended). When all the
 * information has been filled in, the client can commit the changes by calling
 * VNCDiscoverySDKPublishEntity().
 *
 * In the same way, to remove an entity from the address book, the client
 * should use VNCDiscoverySDKUnpublishEntity() and 
 * VNCDiscoverySDKDestroyUnpublishedEntity().
 *
 * The Address Book Discoverer should only be used while it is running.
 *
 * Most Discoverers will not support this Address Book functionality.
 *
 * \subsection subsection_usage_dissapear Disappearing Entities
 *
 * While the Discoverer is still running, it might also report some entities
 * disappearing. This is so that an UI application can keep a list of all the
 * entities that are available.
 *
 * \subsection subsection_usage_closing Stopping and Destroying a Discoverer
 *
 * When all the entity information is retrieved, the client application can
 * choose to stop the Discoverer running through
 * VNCDiscoverySDKDiscovererStop() or VNCDiscoverySDKDiscovererStopEx().
 * A Discoverer can be re-started later on.
 *
 * When a Discoverer is no longer needed, the client can request that the SDK
 * destroys it (and unloads the plug-in), through 
 * VNCDiscoverySDKDiscovererDestroy().
 *
 * \subsection subsection_usage_shutdown Destroying and unloading the SDK
 *
 * -# Call VNCDiscoverySDKDestroy().
 * -# Unload the SDK (necessary only if you did not link against it).
 *    - on Windows, call FreeLibrary()
 *    - on Linux, call dlclose()
 *
 * \section section_memory Memory management
 *
 * The client application owns all the objects it passes to the SDK. 
 *
 * Also the client application owns all the objects passed back from 
 * the SDK. To free memory for anything that has been allocated by the SDK, the
 * client should use VNCDiscoverySDKFreeString() and 
 * VNCDiscoverySDKFreeKeyValPairArray().
 *
 * \section section_thread The Discovery SDK thread
 *
 * The SDK operates in its own thread. This thread is started by the SDK on
 * initialization and is terminated when the SDK gets destroyed.
 *
 * The client can be notified of the new thread starting, or terminating,
 * through the callbacks: VNCDiscoverySDKThreadStartedCallback() and
 * VNCDiscoverySDKThreadEndedCallback(). These callbacks happen within the
 * Discovery SDK thread.
 *
 * The SDK can be called from any thread, but it will process the request
 * within its own thread. This is done by passing a command through a queue
 * that the SDK services. Synchronous requests do not return until they have
 * been serviced by the SDK.
 *
 * To minimise the possibility of thread safety issues occurring within the
 * Discovery SDK, it is advisable to make all API calls from a single thread.
 *
 * All the callbacks coming from the SDK happen in the SDK thread. This means
 * that you should take care to protect against simultaneous access to any
 * application state that is shared between the callbacks and your own thread,
 * e.g. with a mutex.
 *
 * The user of the SDK must obey the following rules to avoid deadlocks:
 *  - Don't call the Discovery SDK API while holding locks.
 *  - Don't block the Discovery SDK thread to wait for for a response from an
 *  application thread.
 *
 * \section section_legal Legal Information
 *
 * Copyright (C) 2011-2018 RealVNC Ltd.  All Rights Reserved.
 *
 * Details of and copyright notices for third-party software that is used by
 * the VNC Discovery SDK can be found in the file Acknowledgements.txt in the SDK
 * distribution.
 *
 * RealVNC and VNC are trademarks of RealVNC Limited and are protected by
 * trademark registrations and/or pending trademark applications in the
 * European Union, United States of America and other jurisdictions.
 *
 * MirrorLink is a registered trademark of Car Connectivity Consortium LLC.
 *
 * Other trademarks are the property of their respective owners.
 *
 * Protected by UK patents 2481870, 2491657; US patents 8760366, 9137657; EU patent 2652951.
 *
 * \see VNCDiscoverySDK, VNCDiscoverySDKInitialize, VNCDiscoverySDKCallbacks
 */

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \brief Associates a context pointer with a Discovery SDK instance.
 *
 * Use this method to associate data with a Discovery SDK instance. The data
 * will be passed back to the client application as a parameter in the
 * callbacks. The client can retrieve the context pointer by calling 
 * VNCDiscoverySDKGetContext().
 *
 * Since some callbacks, like the log callbacks and the thread started
 * callback, can be received before the client application has the chance to
 * set the context, it is recommended that the context is created during
 * initialization. This would be done by setting
 * VNCDiscoverySDKCallbacks::pSDKContext before passing the structure to
 * VNCDiscoverySDKInitialize.
 *
 * \note A second call to VNCDiscoverySDKSetContext() replaces the context 
 *      pointer assigned by the first call.
 * \note The SDK will not interact in any way with the memory to which the
 *      context pointer points.  In particular, the SDK will not free the 
 *      memory when the Discovery SDK instance is destroyed.
 *
 * \param pSDKInstance The SDK instance for which to associate the context.
 * MUST not be NULL (otherwise a crash will occur).
 * \param pSDKContext The new context that is associated with the SDK from now on.
 *          The memory of the old context will not be freed by the SDK. Can be
 *          set to NULL to remove any previous association.
 *
 * \see VNCDiscoverySDKCallbacks
 * \see VNCDiscoverySDKGetContext
 */
typedef void VNCCALL
VNCDiscoverySDKSetContext(VNCDiscoverySDKInstance *pSDKInstance,
                          void *pSDKContext);

/**
 * \brief Retrieves the Discovery SDK instance's context pointer.
 *
 * \param pSDKInstance The SDK instance for which to retrieve the context
 *          pointer. MUST not be NULL (otherwise a crash will occur).
 *
 * \return The pointer to the context data associated with the SDK.
 *
 * \see VNCDiscoverySDKSetContext
 */
typedef void *VNCCALL
VNCDiscoverySDKGetContext(VNCDiscoverySDKInstance *pSDKInstance);

/**
 * \brief Sets the level of logging that will be output by the SDK.
 *
 * The SDK logs at different severity levels. The severity is an integer
 * between 0 and 100. The SDK defines the following severities, but may use
 * values in between:
 *
 *  -   0 - serious errors
 *  -  10 - less serious errors (usually not fatal)
 *  -  20 - warnings
 *  -  30 - informational messages
 *  - 100 - verbose debugging
 *
 * By default the logging level is 0.
 *
 * Once this call returns the SDK will return only logs that correspond to the
 * requested severity.
 *
 * \param pSDKInstance The instance of the SDK for which to set the logging
 *          severity. MUST not be NULL (otherwise a crash will occur).
 * \param severity The maximum severity for which to receive logs (only logs
 *          between 0 and severity will be received).
 *
 * \see VNCDiscoverySDKLogCallback
 */
typedef void VNCCALL
VNCDiscoverySDKSetLoggingSeverity(VNCDiscoverySDKInstance *pSDKInstance,
                                vnc_int32_t severity);

/**
 * \brief Creates a new Discoverer instance of a specific type.
 *
 * Creates a new instance of a VNC Discoverer of the given type. If necessary
 * it will load the specific Discoverer plug-in dll or shared object. 
 *
 * The SDK will use the library with the name vncdiscoverer-type.dll (in
 * Windows), or libvncdiscoverer-type.so (in Linux), where type is the type passed
 * into this call. The client application needs to make sure the library is
 * found in its search path, or provide the pLoadPath argument.
 * 
 * \param pSDKInstance The instance of the SDK that will create the Discoverer.
 *                     The instance is created when initializing the SDK and
 *                     should be kept by the client application for the purpose
 *                     of creating Discoverers.
 * \param pType The string describing the type of Discoverer to load. This is
 *              owned by the client application.
 * \param pLoadPath The path from where to load the Discoverer plug-in. To load
 *                  from the library search path or create a statically 
 *                  registered Discoverer, leave this NULL. 
 * \param pDiscoverer The address of a pointer which on successful loading will
 *                    be pointing to the new Discoverer instance. If the
 *                    creation fails this will point to NULL. It is recommended
 *                    to have this pointing to NULL at this stage.
 *
 * \return VNCDiscoverySDKErrorNone if the Discoverer is created successfully,
 *         VNCDiscoverSDKErrorOutOfMemory if there is not enough memory to
 *         create the new Discoverer, VNCDiscoverySDKErrorInvalidParameter if the
 *         SDK instance passed in, or pType, is NULL,
 *         VNCDiscoverySDKErrorLibraryNotLoaded if the library for the requested 
 *         type can not be loaded correctly, or
 *         VNCDiscoverySDKErrorFeatureNotLicensed if the discoverer requires
 *         a feature not provided by the licenses added to the SDK instance.
 *
 * \see VNCDiscoverySDKInitialize
 * \see VNCDiscoverySDKAddLicense
 * \see VNCDiscoverySDKRegisterDiscoverer
 */
typedef VNCDiscoverySDKError VNCCALL 
VNCDiscoverySDKCreateDiscoverer(VNCDiscoverySDKInstance *pSDKInstance,
                      const char *pType,
                      const char *pLoadPath,
                      VNCDiscoverySDKDiscoverer **pDiscoverer);

/**
 * \brief Associates a context pointer with a Discoverer instance.
 *
 * Use this method to associate data with a Discoverer instance. The client 
 * can retrieve the context pointer by calling 
 * VNCDiscoverySDKDiscovererGetContext().
 *
 * \note A second call to VNCDiscoverySDKDiscovererSetContext() replaces the 
 *      context pointer assigned by the first call.
 * \note The SDK will not interact in any way with the memory to which the
 *      context pointer points.  In particular, the SDK will not free the 
 *      memory when the Discoverer instance is destroyed.
 * \note The Discoverer context will not be passed as a parameter in the
 *      callbacks.
 *
 * \param pDiscoverer The Discoverer for which to set the context. MUST not
 * be NULL (otherwise a crash will occur).
 * \param pDiscovererContext The new context that is associated with the Discoverer from 
 *      now on. The memory of the old context will not be freed by the SDK. Can
 *      be set to NULL to remove any previous association.
 *
 * \see VNCDiscoverySDKDiscovererGetContext
 */
typedef void VNCCALL
VNCDiscoverySDKDiscovererSetContext(VNCDiscoverySDKDiscoverer *pDiscoverer,
                          void *pDiscovererContext);

/**
 * \brief Retrieves the Discoverer instance's context pointer.
 *
 * \param pDiscoverer The Discoverer for which to retrieve the associated
 *          context. MUST not be NULL (otherwise a crash will occur).
 *
 * \return The pointer to the context associated with the Discoverer.
 *
 * \see VNCDiscoverySDKDiscovererSetContext
 */
typedef void *VNCCALL
VNCDiscoverySDKDiscovererGetContext(VNCDiscoverySDKDiscoverer *pDiscoverer);

/**
 * \brief Sets a specific property on a Discoverer instance.
 *
 * This call is synchronous and it sets the property with the given value. If
 * the client wishes to un-set the property it should pass NULL for the value
 * pointer.
 *
 * The properties supported by Discoverers can differ from one type of
 * Discoverer to another. Refer to the Discoverer documentation to understand
 * what properties are supported and what they can be set to.
 *
 * \param pDiscoverer The Discoverer on which to set the property.
 * \param pProperty The string that names the property to be set. This string
 *                  is owned by the client of the SDK.
 * \param pValue The string representing the value to which to set the
 *               property. This can be set to NULL, which means un-set. This
 *               string is owned by the client of the SDK.
 *
 * \return VNCDiscoverySDKErrorNone if the set was successful,
 *         VNCDiscoverySDKErrorNotSupported if the Discoverer doesn't support
 *         this property, VNCDiscoverySDKErrorOutOfMemory if there is not
 *         enough memory to complete the request,
 *         VNCDSicoverySDKErrorInvalidParameter if the Discoverer passed to this
 *         method, or the Property passed to this call is NULL, or another 
 *         Discoverer-specific error code.
 *
 * \see VNCDiscoverySDKError
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKDiscovererSetProperty(VNCDiscoverySDKDiscoverer *pDiscoverer,
                           const char *pProperty,
                           const char *pValue);

/**
 * \brief Gets a specific property from a Discoverer instance.
 *
 * This call is synchronous and it retrieves the value for the specific 
 * property. If the property is not set the return value will be NULL pointer.
 *
 * The properties supported by Discoverers can differ from one type of
 * Discoverer to another. Refer to the Discoverer documentation to understand
 * what properties are supported and what they can be set to.
 *
 * \param pDiscoverer The Discoverer from which to retrieved the property.
 * \param pProperty The string that names the property to be retrieved. This 
 *                  string is owned by the client of the SDK.
 * \param pValue The address of the pointer which will be set to point to the 
 *               string representing the value which was retrieved. This can 
 *               be set to NULL, which means the property is not set. This
 *               string is owned by the client of the SDK on return, and must
 *               be freed by calling VNCDiscoverySDKFreeString() when it is
 *               no longer needed. If an error occurred, this will be NULL.
 *
 * \return VNCDiscoverySDKErrorNone if the set was successful,
 *         VNCDiscoverySDKErrorNotSupported if the Discoverer doesn't support
 *         this property, VNCDiscoverySDKErrorOutOfMemory if there is not
 *         enough memory to complete the request,
 *         VNCDiscoverySDKErrorInvalidParameter if the Discoverer, or Property,
 *         passed to this method is NULL, or another Discoverer-specific error 
 *         code.
 *
 * \see VNCDiscoverySDKError
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKDiscovererGetProperty(VNCDiscoverySDKDiscoverer *pDiscoverer,
                           const char *pProperty,
                           char **pValue);

/**
 * \brief Tells a Discoverer to start discovering.
 * Deprecated by VNCDiscoverySDKDiscovererStartEx.
 *
 * Starts the Discoverer process for a specific Discoverer instance. It reports
 * an outcome asynchronously through the call-backs.
 *
 * A Discoverer must be started to be able to do any operations on its entities.
 *
 * A Discoverer can be re-started.
 *
 * Once the Discoverer is started, any subsequent calls to Start will have no
 * effect, but will issue a VNCDiscoverySDKDiscovererStartedCallback with
 * VNCDiscoverySDKErrorNone.
 *
 * If multiple discoverers are being started then it is recommended to
 * use VNCDiscoverySDKDiscovererSartMultiple() to prevent other
 * discoverers from processing events while a list of discoverers to
 * start is iterated through. This can prevent a discoverer from
 * claiming a device which a discoverer started later also wishes to
 * claim.
 *
 * \param pDiscoverer The Discoverer that is requested to start. MUST not be
 * NULL (otherwise a crash will occur).
 *
 * \see VNCDiscoverySDKCallbacks
 */
typedef void VNCCALL
VNCDiscoverySDKDiscovererStart(VNCDiscoverySDKDiscoverer *pDiscoverer);

/**
 * \brief Tells a Discoverer to start discovering.
 *
 * Starts the Discoverer process for a specific Discoverer instance. It reports
 * an outcome asynchronously through the call-backs.
 *
 * A Discoverer must be started to be able to do any operations on its entities.
 *
 * A Discoverer can be re-started.
 *
 * Once the Discoverer is started, any subsequent calls to Start will have no
 * effect, but will issue a VNCDiscoverySDKDiscovererStartedCallback with
 * VNCDiscoverySDKErrorNone.
 *
 * If multiple discoverers are being started then it is recommended to
 * use VNCDiscoverySDKDiscovererSartMultiple() to prevent other
 * discoverers from processing events while a list of discoverers to
 * start is iterated through. This can prevent a discoverer from
 * claiming a device which a discoverer started later also wishes to
 * claim.
 *
 * \param pDiscoverer The Discoverer that is requested to start. MUST not be
 * NULL (otherwise a crash will occur).
 *
 * \return VNCDiscoverySDKErrorNone if the request is going to be handled.
 *         Other value if the request cannot be handled
 *
 * \see VNCDiscoverySDKError
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKDiscovererStartEx(VNCDiscoverySDKDiscoverer *pDiscoverer);

/**
 * \brief Tells multiple discoverers to start discovering.
 * Deprecated by VNCDiscoverySDKDiscovererStartMultipleEx.
 *
 * Starts the Discoverer process for multiple Discoverer instances. It reports
 * all outcomes asynchronously through the call-backs.
 *
 * A Discoverer must be started to be able to do any operations on its entities.
 *
 * A Discoverer can be re-started.
 *
 * Once the Discoverer is started, any subsequent calls to Start will have no
 * effect, but will issue a VNCDiscoverySDKDiscovererStartedCallback with
 * VNCDiscoverySDKErrorNone.
 *
 * Starting multiple discoverers using this API will ensure that no
 * discoverers can perform processing, such as claiming devices
 * provided by device providers, before all discoverers are started.
 *
 * \param pDiscoverers An array of Discoverers that are requested to
 * start. MUST not be NULL (otherwise a crash will occur).
 * \param discoverersCount The number of items in pDiscoverers
 *
 * \see VNCDiscoverySDKCallbacks
 */
typedef void VNCCALL
VNCDiscoverySDKDiscovererStartMultiple(VNCDiscoverySDKDiscoverer **pDiscoverers,
                                       size_t discoverersCount);

/**
 * \brief Tells multiple discoverers to start discovering.
 *
 * Starts the Discoverer process for multiple Discoverer instances. It reports
 * all outcomes asynchronously through the call-backs.
 *
 * A Discoverer must be started to be able to do any operations on its entities.
 *
 * A Discoverer can be re-started.
 *
 * Once the Discoverer is started, any subsequent calls to Start will have no
 * effect, but will issue a VNCDiscoverySDKDiscovererStartedCallback with
 * VNCDiscoverySDKErrorNone.
 *
 * Starting multiple discoverers using this API will ensure that no
 * discoverers can perform processing, such as claiming devices
 * provided by device providers, before all discoverers are started.
 *
 * \param pDiscoverers An array of Discoverers that are requested to
 * start. MUST not be NULL (otherwise a crash will occur).
 * \param discoverersCount The number of items in pDiscoverers
 *
 * \return VNCDiscoverySDKErrorNone if the request is going to be handled.
 *         Other value if the request cannot be handled
 *
 * \see VNCDiscoverySDKError
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKDiscovererStartMultipleEx(VNCDiscoverySDKDiscoverer **pDiscoverers,
                                       size_t discoverersCount);
/**
 * \brief Tells a Discoverer to stop discovering.
 * Deprecated by VNCDiscoverySDKDiscovererStopEx.
 *
 * Stops the Discoverer process for a specific Discoverer instance.
 *
 * The Discoverer will confirm through a call back that the stop was
 * successful. It is possible for individual Discoverers to report
 * problems occuring while trying to stop.
 *
 * Any calls on entities discovered by the instance can fail after this
 * call. This is because entities can disappear while the
 * Discoverer is stopped. The only operations that will still work
 * for entities are the get/set context data.
 *
 * Any active requests on the Discoverer will fail with
 * VNCDiscoverySDKErrorCancelled.
 *
 * A Discoverer can be re-started.
 *
 * Once the Discoverer is stopped, any subsequent calls to Stop will have no
 * effect, but will issue a VNCDiscoverySDKDiscovererStoppedCallback with
 * VNCDiscoverySDKErrorNone.
 *
 * If multiple discoverers are being stopped then it is recommended to
 * use VNCDiscoverySDKDiscovererStopMultiple() or
 * VNCDiscoverySDKDiscovererStopMultipleEx() to prevent other
 * discoverers from processing events while a list of discoverers to
 * stop is iterated through.
 *
 * \param pDiscoverer The Discoverer that is requested to stop. MUST not be
 * NULL (otherwise a crash will occur).
 *
 * \see VNCDiscoverySDKCallbacks
 */
typedef void VNCCALL
VNCDiscoverySDKDiscovererStop(VNCDiscoverySDKDiscoverer *pDiscoverer);

/**
 * \brief Tells a Discoverer to stop discovering.
 *
 * Stops the Discoverer process for a specific Discoverer instance.
 *
 * The Discoverer will confirm through a call back that the stop was
 * successful. It is possible for individual Discoverers to report
 * problems occuring while trying to stop.
 *
 * Any calls on entities discovered by the instance can fail after this
 * call. This is because entities can disappear while the
 * Discoverer is stopped. The only operations that will still work
 * for entities are the get/set context data.
 *
 * Any active requests on the Discoverer will fail with
 * VNCDiscoverySDKErrorCancelled.
 *
 * A Discoverer can be re-started.
 *
 * Once the Discoverer is stopped, any subsequent calls to Stop will have no
 * effect, but will issue a VNCDiscoverySDKDiscovererStoppedCallback with
 * VNCDiscoverySDKErrorNone.
 *
 * If multiple discoverers are being stopped then it is recommended to
 * use VNCDiscoverySDKDiscovererStopMultiple() or
 * VNCDiscoverySDKDiscovererStopMultipleEx() to prevent other
 * discoverers from processing events while a list of discoverers to
 * stop is iterated through.
 *
 * \param pDiscoverer The Discoverer that is requested to stop. MUST not be
 * NULL (otherwise a crash will occur).
 *
 * \return VNCDiscoverySDKErrorNone if the request is going to be handled.
 *         Other value if the request cannot be handled
 *
 * \see VNCDiscoverySDKError
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKDiscovererStopEx(VNCDiscoverySDKDiscoverer *pDiscoverer);

/**
 * \brief Tells multiple discoverers to stop discovering.
 * Deprecated by VNCDiscoverySDKDiscovererStopMultipleEx.
 *
 * Stops the Discoverer process for a specific list of Discoverer instances.
 *
 * The Discoverers will confirm through a call back that the stop was
 * successful. It is possible for individual Discoverers to report
 * problems occuring while trying to stop.
 *
 * Any calls on entities discovered by the instance can fail after this
 * call. This is because entities can disappear while the
 * Discoverer is stopped. The only operations that will still work
 * for entities are the get/set context data.
 *
 * Any active requests on the Discoverer will fail with
 * VNCDiscoverySDKErrorCancelled.
 *
 * A Discoverer can be re-started.
 *
 * Once the Discoverer is stopped, any subsequent calls to Stop will have no
 * effect, but will issue a VNCDiscoverySDKDiscovererStoppedCallback with
 * VNCDiscoverySDKErrorNone.
 *
 * Stopping multiple discoverers using this API will ensure that no
 * discoverers can perform processing, such as claiming devices
 * provided by device providers, between discoverers being asked to stop.
 *
 * \param pDiscoverers An array of Discoverers that are requested to
 * stop. MUST not be NULL (otherwise a crash will occur).
 * \param discoverersCount The number of items in pDiscoverers
 *
 * \see VNCDiscoverySDKCallbacks
 */
typedef void VNCCALL
VNCDiscoverySDKDiscovererStopMultiple(VNCDiscoverySDKDiscoverer **pDiscoverers,
                                       size_t discoverersCount);

/**
 * \brief Tells multiple discoverers to stop discovering.
 *
 * Stops the Discoverer process for a specific list of Discoverer instances.
 *
 * The Discoverers will confirm through a call back that the stop was
 * successful. It is possible for individual Discoverers to report
 * problems occuring while trying to stop.
 *
 * Any calls on entities discovered by the instance can fail after this
 * call. This is because entities can disappear while the
 * Discoverer is stopped. The only operations that will still work
 * for entities are the get/set context data.
 *
 * Any active requests on the Discoverer will fail with
 * VNCDiscoverySDKErrorCancelled.
 *
 * A Discoverer can be re-started.
 *
 * Once the Discoverer is stopped, any subsequent calls to Stop will have no
 * effect, but will issue a VNCDiscoverySDKDiscovererStoppedCallback with
 * VNCDiscoverySDKErrorNone.
 *
 * Stopping multiple discoverers using this API will ensure that no
 * discoverers can perform processing, such as claiming devices
 * provided by device providers, between discoverers being asked to stop.
 *
 * \param pDiscoverers An array of Discoverers that are requested to
 * stop. MUST not be NULL (otherwise a crash will occur).
 * \param discoverersCount The number of items in pDiscoverers
 *
 * \return VNCDiscoverySDKErrorNone if the request is going to be handled.
 *         Other value if the request cannot be handled
 *
 * \see VNCDiscoverySDKError
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKDiscovererStopMultipleEx(VNCDiscoverySDKDiscoverer **pDiscoverers,
                                       size_t discoverersCount);

/**
 * \brief  Notifies the Discovery SDK that it can now free the resources
 *         associated with the specified entity
 *
 * This call destroys an Entity from memory.
 *
 * This is an asynchronous call.
 *
 * If the 'AcknowledgeEntityDisappeared' property is set and this call is not made the Entity will live
 * in memory until the Discoverer is destroyed.
 *
 * \param pDiscoverer The Discoverer that owns the entity.
 * \param pEntity The entity that should be destroyed.
 *
 */
typedef void VNCCALL
VNCDiscoverySDKAcknowledgeEntityDisappeared(VNCDiscoverySDKDiscoverer *pDiscoverer,
                                  VNCDiscoverySDKEntity *pEntity);

/**
 * \brief Retrieves the Viewer Parameters needed in addition to vnccmd.
 *
 * Gets an array of the Viewer Parameters (as key-value pairs) for the 
 * Entity. This is a synchronous call.
 *
 * The Viewer Parameters are needed because sometimes the vnccmd is not
 * sufficient to establish a connection to the VNC Server.
 *
 * \param pDiscoverer The Discoverer that owns the entity.
 * \param pEntity The Entity for which to retrieve the Viewer Parameters.
 * \param pParametersArray The address of the pointer which will point to the
 *                         array of key-value pairs on return from this method.
 *                         If some error occurred, or there are no parameters
 *                         needed this will be NULL. This array is owned by the
 *                         client of the SDK, and must be freed by calling
 *                         VNCDiscoverySDKFreeKeyValPairArray when it is no
 *                         longer needed.
 * \param pParametersArraySize The address where the Discoverer should write
 *                             the size of the returned array.
 *
 * \return VNCDiscoverySDKErrorNone if everything was well,
 *         VNCDiscoverySDKErrorOutOfMemory if there is not enough memory to complete
 *         the request, VNCDiscoverySDKErrorNotFound if the entity can't be
 *         found, or another Discoverer-specific error.
 *
 * \see VNCDiscoverySDKError
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKGetViewerParameters(VNCDiscoverySDKDiscoverer *pDiscoverer,
                            VNCDiscoverySDKEntity *pEntity, 
                            VNCDiscoverySDKKeyValPair **pParametersArray,
                            size_t *pParametersArraySize); 

/**
 * \brief Fetches the value for a key of an entity.
 *
 * Initiates an asynchronous fetch of a specific Entity Value, for
 * a given Discoverer and Entity, based on its key. This call is asynchronous
 * and the value will be passed back through a call-back.
 *
 * If there is an error occuring while making this request, this will be
 * reported through the callback.
 *
 * There are 4 keys that all Discoverers must support: "id", "type", "name" and
 * "vnccmd". On top of those there can be some Discoverer-specific keys. Please
 * refer to the Discoverer documentation for those.
 *
 * When a fetch request is made the client application receives an ID of the
 * request. It can then use the ID from the callback to match the request.
 *
 * \note If there is an error in completing the request the SDK will
 * make a callback specifying the error.
 *
 * \note It can happen that the callback with the result is made before this
 * method completes. (For example if this call is made from the SDK thread and
 * the Discoverer responds straight away, from the SDK thread as well).
 *
 * \note The existence of an entity is not definite, even if a vnccmd could be
 * retrieved successfully. It is not guaranteed that a VNC Server is present 
 * and listening for incoming connections. False positives might be reported by 
 * some Discoverers.
 *
 * \note The callback from this request will go only to the main callbacks. If
 * an alternate listener should receive the response, then use
 * VNCDiscoverySDKFetchEntityValueRespondToListener.
 *
 * \param pDiscoverer The Discoverer that owns the entity. MUST not be NULL
 * (otherwise a crash will occur).
 * \param pEntity The Entity from which to read the value.
 * \param pKey The key for the value to be read. This is a string owned by the
 *             client of the SDK.
 * \param timeOut The number of microseconds to wait before timing out the
 *                request. If you don't need a time out, then set this to
 *                VNCDiscoverySDKNoTimeout. If you want the call to be instant,
 *                so no waiting at all, set it to VNCDiscoverySDKInstantCall.
 *
 * \return An ID of the request which will be reported in the callback. If
 *         there is some error initiating the request this ID will be set to
 *         VNCDiscoverySDKNoRequestId. In this case no callback will be made.
 *         Passing in a NULL entity or key results in
 *         a VNCDiscoverySDKNoRequestId.
 *
 * \see VNCDiscoverySDKCallbacks, VNCDiscoverySDKTimeout
 * \see VNCDiscoverySDKFetchEntityValueCallback
 * \see VNCDiscoverySDKFetchEntityValueRespondToListener
 */
typedef VNCDiscoverySDKRequestId VNCCALL
VNCDiscoverySDKFetchEntityValue(VNCDiscoverySDKDiscoverer *pDiscoverer,
                              VNCDiscoverySDKEntity *pEntity,
                              const char *pKey,
                              VNCDiscoverySDKTimeoutMicroseconds timeOut);

/**
 * \brief Gets the value for a key of an entity synchronously.
 *
 * Fetches synchronously a specific Entity Value, for a given Discoverer and 
 * Entity, based on its key. 
 * 
 * For the keys "id" and "type" the fetch is guaranteed to be quick, but other 
 * properties might be retrieved very slowly. So avoid using this call for the
 * other keys, unless you know from the Discoverer documentation that the fetch
 * is quick. If you are not sure, but want to try and use this, make sure you
 * set a timeout for the request.
 *
 * \param pDiscoverer The Discoverer that owns the entity.
 * \param pEntity The Entity from which to read the value.
 * \param pKey The key for the value to be read. This is a string owned by the
 *             client of the SDK.
 * \param timeOut The number of microseconds to wait before timing out the
 *                request. If you don't need a time out, then set this to
 *                VNCDiscoverySDKNoTimeout. If you want the call to be instant,
 *                so no waiting at all, set it to VNCDiscoverySDKInstantCall.
 *                request.
 * \param pResultValue The address of the pointer which will point to the
 *                     result value string. This will be owned by the client of
 *                     the SDK, and must be freed by calling
 *                     VNCDiscoverySDKFreeString() when it is no longer needed.
 *                     If there is any problem when retrieving the value the
 *                     pointer will be NULL.
 *
 * \return VNCDiscoverySDKErrorNone if the request was completed successfully,
 *         VNCDiscoverySDKErrorNotSupported if the key is not supported by the
 *         Entity, VNCDiscoverySDKErrorOutOfMemory if there is not enough memory to
 *         complete the request, VNCDiscoverySDKErrorInvalidParameter if the 
 *         Entity, or Key, are NULL, VNCDiscoverySDKErrorUnknownEntity 
 *         if the Entity doesn't exist, VNCDiscoverySDKErrorCancelled if the
 *         Discoverer was stopped while processing the request, or another error specific to the 
 *         Discoverer.
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKFetchEntityValueBlocking(VNCDiscoverySDKDiscoverer *pDiscoverer,
                              VNCDiscoverySDKEntity *pEntity,
                              const char *pKey,
                              char **pResultValue,
                              VNCDiscoverySDKTimeoutMicroseconds timeOut);

/**
 * \brief Gets some binary data owned by an entity.
 *
 * This method asks the Discoverer to get a pointer to some binary data owned
 * by the entity. To identify the data a key is used. This key might have been
 * passed as a result of a Fetch Entity Value, or be a documented key.
 *
 * Check the Discoverer specific documentation to see if there is any binary
 * data that can be retrieved, and how to get the key for the data.
 *
 * If the binary data can't be found then the return size will be 0 and the
 * pointer will point to NULL.
 *
 * \param pDiscoverer The Discoverer that owns the entity. MUST not be NULL
 * (otherwise a crash will occur).
 * \param pEntity The Entity from which to retrieve the binary data.
 * \param pKey The key of the data. This key is either something documented by
 *          the Discoverer, or retrieved from the entity as a result of a Fetch
 *          Entity Value.
 * \param pData This will point to the binary data on return, or NULL if the
 *          data can't be found, or either the entity or the key is NULL.
 *
 * \return The size of the binary data. If the data can't be found the size is
 *          0.
 */
typedef size_t VNCCALL
VNCDiscoverySDKGetEntityBinaryData(VNCDiscoverySDKDiscoverer *pDiscoverer,
                              VNCDiscoverySDKEntity *pEntity,
                              const char *pKey,
                              const vnc_uint8_t **pData);

/**
 * \brief Posts a value to a key of an entity.
 *
 * Asynchronously posts a new value for a given key of an Entity.
 *
 * It is up to the Discoverer to decide how to handle the post command. Some
 * might treat all keys as read-only, others only some of them, while others
 * might allow posting to any key (for example an addressbook style
 * Discoverer).
 *
 * The key-value pair is formed by strings.
 * 
 * This request can only be made while the Discoverer is still running.
 *
 * \note If there is an error in completing the request the SDK will
 * make a callback specifying the error.
 *
 * \note It can happen that the callback with the result is made before this
 * method completes. (For example if this call is made from the SDK thread and
 * the Discoverer responds straight away, from the SDK thread as well).
 *
 * \note The callback from this request will go only to the main callbacks. If
 * an alternate listener should receive the response, then use
 * VNCDiscoverySDKPostEntityValueRespondToListener.
 *
 * \param pDiscoverer The Discoverer that owns the entity. MUST not be NULL
 * (otherwise a crash will occur).
 * \param pEntity The Entity to which to set the value.
 * \param pKey The key for which to set the new value. This string is owned by
 *             the client of the SDK.
 * \param pValue The value to which to set the key. This value can be NULL,
 *               meaning that the key is to be un-set. This string is owned by
 *               the client of the SDK.
 * \param timeOut The number of microseconds to wait before timing out the
 *                request. If you don't need a time out, then set this to
 *                VNCDiscoverySDKNoTimeout. If you want the call to be instant,
 *                so no waiting at all, set it to VNCDiscoverySDKInstantCall.
 *
 * \return An ID of the request which will be reported in the callback. If
 *         there is some error making the request this ID will be set to
 *         VNCDiscoverySDKNoRequestId. Passing in a NULL entity or key results
 *         in a VNCDiscoverySDKNoRequestId.
 *
 * \see VNCDiscoverySDKCallbacks, VNCDiscoverySDKTimeout
 * \see VNCDiscoverySDKPostEntityValueRespondToListener
 */
typedef VNCDiscoverySDKRequestId VNCCALL
VNCDiscoverySDKPostEntityValue(VNCDiscoverySDKDiscoverer *pDiscoverer,
                              VNCDiscoverySDKEntity *pEntity,
                              const char *pKey,
                              const char *pValue,
                              VNCDiscoverySDKTimeoutMicroseconds timeOut);

/**
 * \brief Posts a value synchronously to a key of an entity.
 *
 * Asynchronously posts a new value for a given key of an Entity.
 *
 * It is up to the Discoverer to decide how to handle the post command. Some
 * might treat all keys as read-only, others only some of them, while others
 * might allow posting to any key (for example an addressbook style
 * Discoverer).
 *
 * The key-value pair is formed by strings.
 *
 * This request can only be made while the Discoverer is still running.
 *
 * \param pDiscoverer The Discoverer that owns the entity.
 * \param pEntity The Entity to which to set the value.
 * \param pKey The key for which to set the new value. This string is owned by
 *             the client of the SDK.
 * \param pValue The value to which to set the key. This value can be NULL,
 *               meaning that the key is to be un-set. This string is owned by
 *               the client of the SDK.
 * \param timeOut The number of microseconds to wait before timing out the
 *                request. If you don't need a time out, then set this to
 *                VNCDiscoverySDKNoTimeout. If you want the call to be instant,
 *                so no waiting at all, set it to VNCDiscoverySDKInstantCall.
 *
 * \return VNCDiscoverySDKErrorNone if the request was completed successfully,
 *         VNCDiscoverySDKErrorNotSupported if the key is not supported, or
 *         read-only by the Entity, VNCDiscoverySDKErrorOutOfMemory if there 
 *         is not enough memory to complete the request, 
 *         VNCDiscoverySDKErrorInvalidParameter if the Discoverer, Entity, 
 *         or Key, is NULL, or pValue is NULL and the Entity doesn't support
 *         un-setting of the key, VNCDiscoverySDKErrorUnknownEntity if the Entity 
 *         doesn't exist, VNCDiscoverySDKErrorCancelled if the Discoverer was
 *         stopped while processing the request, or another error specific to the Discoverer.
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKPostEntityValueBlocking(VNCDiscoverySDKDiscoverer *pDiscoverer,
                              VNCDiscoverySDKEntity *pEntity,
                              const char *pKey,
                              const char *pValue,
                              VNCDiscoverySDKTimeoutMicroseconds timeOut);

/**
 * \brief Associates a context pointer with an Entity.
 *
 * This call sets the context data associated with an Entity.
 *
 * The context data can be retrieved at any point during the lifetime of the 
 * Entity calling VNCDiscoverySDKEntityGetContext().
 *
 * \note A second call to VNCDiscoverySDKEntitySetContext() replaces the 
 *      context pointer assigned by the first call.
 * \note The SDK will not interact in any way with the memory to which the
 *      context pointer points.  In particular, the SDK will not free the 
 *      memory when the Entity is destroyed.
 * \note The Entity context will not be passed as a parameter in the callbacks.
 * 
 * \param pDiscoverer The Discoverer that owns the entity. MUST not be NULL
 * (otherwise a crash will occur).
 * \param pEntity The Entity that is associated with the context data. MUST not
 * be NULL (otherwise a crash will occur).
 * \param pEntityContext The context data that is associated with the entity. Can be 
 *          NULL, in which case it indicates that the entity should clear the 
 *          context data.
 *
 * \see VNCDiscoverySDKEntityGetContext
 */
typedef void VNCCALL
VNCDiscoverySDKEntitySetContext(VNCDiscoverySDKDiscoverer *pDiscoverer,
                                  VNCDiscoverySDKEntity *pEntity,
                                  void *pEntityContext);

/**
 * \brief Retrieve the Entity's context pointer.
 *
 * This call returns the context data that has been associated with the Entity.
 * The call can be made at any time during the lifetime of the Entity, whether 
 * the Discoverer is running or not. The last time this can be called is during
 * the Entity Disappeared call, but after that the context data won't be
 * retrieved anymore.
 *
 * \param pDiscoverer The Discoverer that owns the entity. MUST not be NULL
 * (otherwise a crash will occur).
 * \param pEntity The Entity that holds the context data. MUST not be NULL
 * (otherwise a crash will occur).
 *
 * \return A pointer to the context data. This is NULL if the context data has
 *         not been set, or set to NULL, or if the entity doesn't exist.
 *
 * \see VNCDiscoverySDKEntitySetContext
 */
typedef void *VNCCALL
VNCDiscoverySDKEntityGetContext(VNCDiscoverySDKDiscoverer *pDiscoverer,
                                  VNCDiscoverySDKEntity *pEntity);

/**
 * \brief Creates an entity to be published by an address book Discoverer.
 *
 * This call creates an Entity in memory which can be published later by an
 * address book Discoverer. 
 *
 * Most Discoverers will not support this call, only those that support address
 * book-like requests.
 *
 * This is a synchronous call.
 *
 * Once an entity is created it can be populated by using one of the Post
 * Entity Value methods (either the synchronous or the asynchronous ones).
 *
 * \param pDiscoverer The Discoverer that should create the entity.
 * \param pEntity The address of the pointer that should point to the newly
 *                created entity. On failure the pointer is NULL. Ownership of
 *                the entity rests with the Discoverer.
 *
 * \return VNCDiscoverySDKErrorNone if the entity was created successfully.
 *         VNCDiscoverySDKErrorNotSupported if the Discoverer doesn't support
 *         Create Entity. VNCDiscoverySDKErrorInvalidParameter if the
 *         Discoverer passed in is NULL. VNCDiscoverySDKOutOfMemory if there is
 *         not enough memory to create the entity. Or another
 *         Discoverer-specific error code.
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKCreateEntity(VNCDiscoverySDKDiscoverer *pDiscoverer,
                          VNCDiscoverySDKEntity **pEntity);

/**
 * \brief Publishes an entity by an address book Discoverer.
 *
 * This call publishes an Entity which has been previously created and
 * populated in memory by an address book Discoverer
 *
 * Most Discoverers will not support this call, only those that support address
 * book-like requests.
 *
 * This is an asynchronous call. The outcome of the operation will be reported
 * through a callback.
 *
 * \param pDiscoverer The Discoverer that owns the entity. MUST not be NULL
 * (otherwise a crash will occur).
 * \param pEntity The entity that should be published. MUST not be NULL
 * (otherwise a crash will occur).
 *
 * \see VNCDiscoverySDKCallbacks
 */
typedef void VNCCALL
VNCDiscoverySDKPublishEntity(VNCDiscoverySDKDiscoverer *pDiscoverer,
                                  VNCDiscoverySDKEntity *pEntity);
 
/**
 * \brief Unpublishes (removes) an entity by an address book Discoverer.
 *
 * This call unpublishes an Entity which has been previously published
 * by an address book Discoverer
 *
 * Most Discoverers will not support this call, only those that support address
 * book-like requests.
 *
 * This is an asynchronous call. The outcome of the operation will be reported
 * through a callback.
 *
 * After this call the entity will still live in memory so the client SDK
 * should call VNCDiscoverySDKDestroyUnpublishedEntity() to ensure freeing of
 * the memory.
 *
 * \param pDiscoverer The Discoverer that owns the entity. MUST not be NULL
 * (otherwise a crash will occur).
 * \param pEntity The entity that should be unpublished. MUST not be NULL
 * (otherwise a crash will occur).
 *
 * \see VNCDiscoverySDKCallbacks
 */
typedef void VNCCALL
VNCDiscoverySDKUnpublishEntity(VNCDiscoverySDKDiscoverer *pDiscoverer,
                                  VNCDiscoverySDKEntity *pEntity);

/**
 * \brief Destroys an entity previously created to be published by an address 
 *        book Discoverer.
 *
 * This call destroys an Entity from memory which was created for publishing 
 * by an address book Discoverer. 
 *
 * Most Discoverers will not support this call, only those that support address
 * book-like requests.
 *
 * This is a synchronous call.
 *
 * This call can only be made if the entity is not published at present.
 *
 * If this call is not made the Entity will live in memory until the Discoverer
 * is destroyed.
 *
 * \param pDiscoverer The Discoverer that owns the entity.
 * \param pEntity The entity that should be destroyed.
 *
 * \return VNCDiscoverySDKErrorNone if the entity was destroyed successfully.
 *         VNCDiscoverySDKErrorNotSupported if the Discoverer doesn't support
 *         Destroy Entity. VNCDiscoverySDKErrorInvalidParameter if the
 *         Discoverer, or the Entity passed in is NULL. Or another
 *         Discoverer-specific error code.
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKDestroyUnpublishedEntity(VNCDiscoverySDKDiscoverer *pDiscoverer,
                                  VNCDiscoverySDKEntity *pEntity);

/**
 * \brief Destroys a Discoverer.
 *
 * Destroys a specific Discoverer instance. All entities found by the
 * Discoverer become invalid as the Discoverer frees all the memory that it
 * held for them.
 * 
 * The client application must make this call for each Discoverer it created,
 * otherwise the resources owned by the Discoverer will not be freed.
 * VNCDiscoverySDKDestroy destroys only the SDK, without destroying the
 * Discoverers themselves.
 *
 * \param pDiscoverer The Discoverer to be destroyed. MUST not be NULL
 * (otherwise a crash will occur).
 */
typedef void VNCCALL
VNCDiscoverySDKDiscovererDestroy(VNCDiscoverySDKDiscoverer *pDiscoverer);

/**
 * \brief Destroys the SDK.
 *
 * This call destroys the SDK so it can't be used by the client anymore after
 * this call, unless the client re-initializes it.
 *
 * This call needs to be made by the client, whenever it finished with the SDK,
 * otherwise the resources will not get freed.
 *
 * \note Calling VNCDiscoverySDKDestroy does not destroy the Discoverers, it
 * only destroys the SDK. Therefore VNCDiscoverySDKDiscovererDestroy must be
 * called for each Discoverer that has been created successfully before making
 * a call to this function.
 *
 * \param pSDKInstance The pointer to the SDK instance to be destroyed. MUST
 * not be NULL (otherwise a crash will occur).
 */
typedef void VNCCALL
VNCDiscoverySDKDestroy(VNCDiscoverySDKInstance *pSDKInstance);

/**
 * \brief Allocates the memory for a string.
 *
 * This method should be used if the client of the SDK has to create a string 
 * and pass its ownership to the SDK. The method can allocate a new string 
 * that is a copy of the original string, or can just allocate a certain 
 * amount of memory.
 * 
 * \param pString The original string. If the new string should be a copy of an
 *                original one this is where the original is specified. If the 
 *                new string is empty then this should be NULL
 * \param length The length of the string. If the first parameter is NULL then
 *               the length should be greater than 0. Otherwise, if there is an
 *               original string, the length specifies how much of the string
 *               to copy. Use -1 if you want the entire string copied.
 *
 * \return A pointer to the new string, or NULL if there is not enough memory
 *        (or NULL was passed initially together with -1 to allocString).
 */
typedef char *VNCCALL
VNCDiscoverySDKAllocString(const char *pString, size_t length);

/**
 * \brief Frees the memory of a string allocated by the SDK.
 *
 * This is used to free any string that was allocated through the SDK, and then
 * passed to the client application.
 *
 * The provided string must be NUL terminated if non-NULL.
 *
 * \param pString The string to be freed.
 */
typedef void VNCCALL
VNCDiscoverySDKFreeString(char *pString);

/**
 * \brief Frees the memory of a key-value array allocated by the SDK.
 *
 * This is used to free the memory occupied by a key-value array which was
 * allocated by the SDK, and then passed to the client application.
 *
 * This method also frees the memory occupied by the strings within the array.
 *
 * \param pArray The array to be freed.
 * \param size The size of the array to be freed
 */
typedef void VNCCALL
VNCDiscoverySDKFreeKeyValPairArray(VNCDiscoverySDKKeyValPair *pArray, size_t size);

/**
 * \brief Returns the current status of a Discoverer.
 *
 * Returns whether the Discoverer is waiting, or scanning. If the Discoverer is
 * not started then the status is not defined.
 *
 * \param pDiscoverer The Discoverer for which to get the status. MUST not be
 * NULL (otherwise a crash will occur).
 *
 * \return The current status of the Discoverer.
 *
 * \see VNCDiscoverySDKDiscovererStatusUpdateCallback
 */
typedef VNCDiscoverySDKDiscovererStatus VNCCALL
VNCDiscoverySDKGetDiscovererStatus(VNCDiscoverySDKDiscoverer *pDiscoverer);

/**
 * \brief Adds a new set of callbacks for a Discoverer.
 *
 * This adds a new set of callbacks for a Discoverer. These will not replace
 * the default ones, but any notifications from the Discoverer will be sent to
 * these callbacks as well.
 *
 * Responses to requests (like VNCDiscoverySDKFetchEntityValue, or
 * VNCDiscoverySDKPostEntityValue) will be sent only to the default 
 * callbacks. If a listener is making a Fetch
 * or Post request and wishes to know the result, then it should use
 * VNCDiscoverySDKFetchEntityValueRespondToListener and
 * VNCDiscoverySDKPostEntityValueRespondToListener.
 *
 * The listener will not receive any logs. All the logs are sent only to the
 * default callbacks for the SDK.
 *
 * The listener should implement ::VNCDiscoverySDKDiscovererDestroyedCallback
 * and if that is called stop using the Discoverer.
 *
 * \param pDiscoverer The Discoverer for which to add a listener.
 * \param pListenerContext The context of the listener that will be used in the
 * callbacks.
 * \param pCallbacks The callback structure for the new listener. Callbacks
 * that are not needed should be left NULL.
 * \param callbacksSize The size of the callbacks structure.
 * \param listenerId Pointer to where the listenerId will be written to when
 * successful.
 *
 * \return VNCDiscoverySDKErrorNone if the listener has been added correctly.
 * Other error codes to indicate the failure reason.
 *
 * \see VNCDiscoverySDKFetchEntityValueRespondToListener
 * \see VNCDiscoverySDKPostEntityValueRespondToListener
 * \see VNCDiscoverySDKRemoveDiscovererListener
 * \see ::VNCDiscoverySDKDiscovererDestroyedCallback
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKAddDiscovererListener(VNCDiscoverySDKDiscoverer *pDiscoverer,
                      void *pListenerContext,
                      VNCDiscoverySDKCallbacks *pCallbacks,
                      size_t callbacksSize,
                      vnc_uint32_t *listenerId);

/**
 * \brief Removes a listener from a Discoverer.
 *
 * This call removes an existing listener from a Discoverer. Any outstanding
 * requests for the listener will receive VNCDiscoverySDKErrorCancelled.
 *
 * \param pDiscoverer The Discoverer from which to remove the listener.
 * \param listenerId The ID of the listener to remove.
 *
 * \return VNCDiscoverySDKErrorNone if the listener has been removed.
 * Other error codes to indicate the failure reason.
 *
 * \see VNCDiscoverySDKAddDiscovererListener
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKRemoveDiscovererListener(VNCDiscoverySDKDiscoverer *pDiscoverer,
                      vnc_uint32_t listenerId);

/**
 * \brief Fetches the value for a key of an entity and sends the response to a
 * particular listener.
 *
 * Initiates an asynchronous fetch of a specific Entity Value, for
 * a given Discoverer and Entity, based on its key. This call is asynchronous
 * and the value will be passed back through a call-back to the listener
 * specified in the the listenerId argument.
 *
 * If there is an error occuring while making this request, this will be
 * reported through the callback.
 *
 * There are 4 keys that all Discoverers must support: "id", "type", "name" and
 * "vnccmd". On top of those there can be some Discoverer-specific keys. Please
 * refer to the Discoverer documentation for those.
 *
 * When a fetch request is made the client application receives an ID of the
 * request. It can then use the ID from the callback to match the request.
 *
 * \note If the listenerId is not valid then no request ID will be given.
 *
 * \note If there is an error in completing the request the SDK will
 * make a callback specifying the error.
 *
 * \note It can happen that the callback with the result is made before this
 * method completes. (For example if this call is made from the SDK thread and
 * the Discoverer responds straight away, from the SDK thread as well).
 *
 * \note The existence of an entity is not definite, even if a vnccmd could be
 * retrieved successfully. It is not guaranteed that a VNC Server is present 
 * and listening for incoming connections. False positives might be reported by 
 * some Discoverers.
 *
 * \param listenerId The listener that is waiting for the response.
 * \param pDiscoverer The Discoverer that owns the entity. MUST not be NULL
 * (otherwise a crash will occur).
 * \param pEntity The Entity from which to read the value.
 * \param pKey The key for the value to be read. This is a string owned by the
 *             client of the SDK.
 * \param timeOut The number of microseconds to wait before timing out the
 *                request. If you don't need a time out, then set this to
 *                VNCDiscoverySDKNoTimeout. If you want the call to be instant,
 *                so no waiting at all, set it to VNCDiscoverySDKInstantCall.
 *
 * \return An ID of the request which will be reported in the callback. If
 *         there is some error initiating the request this ID will be set to
 *         VNCDiscoverySDKNoRequestId. In this case no callback will be made.
 *         If either the entity or key is NULL, or the listener ID is not
 *         valid, then VNCDiscoverySDKNoRequestId will be returned.
 *
 * \see VNCDiscoverySDKCallbacks, VNCDiscoverySDKTimeout
 * \see VNCDiscoverySDKFetchEntityValueCallback
 * \see VNCDiscoverySDKFetchEntityValue
 * \see VNCDiscoverySDKAddDiscovererListener
 */
typedef VNCDiscoverySDKRequestId VNCCALL
VNCDiscoverySDKFetchEntityValueRespondToListener(
                              vnc_uint32_t listenerId,
                              VNCDiscoverySDKDiscoverer *pDiscoverer,
                              VNCDiscoverySDKEntity *pEntity,
                              const char *pKey,
                              VNCDiscoverySDKTimeoutMicroseconds timeOut);

/**
 * \brief Posts a value to a key of an entity and reports the outcome to a
 * particular listener.
 *
 * Asynchronously posts a new value for a given key of an Entity. The outcome
 * of the request will be reported through a call-back to the listener
 * specified by the listenerId argument.
 *
 * It is up to the Discoverer to decide how to handle the post command. Some
 * might treat all keys as read-only, others only some of them, while others
 * might allow posting to any key (for example an addressbook style
 * Discoverer).
 *
 * The key-value pair is formed by strings.
 * 
 * This request can only be made while the Discoverer is still running.
 *
 * \note If the listenerId is not valid then no request ID will be given.
 *
 * \note If there is an error in completing the request the SDK will
 * make a callback specifying the error.
 *
 * \note It can happen that the callback with the result is made before this
 * method completes. (For example if this call is made from the SDK thread and
 * the Discoverer responds straight away, from the SDK thread as well).
 *
 * \param listenerId The listener that is waiting for the response.
 * \param pDiscoverer The Discoverer that owns the entity.
 * \param pEntity The Entity to which to set the value.
 * \param pKey The key for which to set the new value. This string is owned by
 *             the client of the SDK.
 * \param pValue The value to which to set the key. This value can be NULL,
 *               meaning that the key is to be un-set. This string is owned by
 *               the client of the SDK.
 * \param timeOut The number of microseconds to wait before timing out the
 *                request. If you don't need a time out, then set this to
 *                VNCDiscoverySDKNoTimeout. If you want the call to be instant,
 *                so no waiting at all, set it to VNCDiscoverySDKInstantCall.
 *
 * \return An ID of the request which will be reported in the callback. If
 *         there is some error making the request this ID will be set to
 *         VNCDiscoverySDKNoRequestId. If either the entity or key is NULL, or
 *         the listener ID is not valid, then VNCDiscoverySDKNoRequestId will
 *         be returned.
 *
 * \see VNCDiscoverySDKCallbacks, VNCDiscoverySDKTimeout
 * \see VNCDiscoverySDKPostEntityValue
 * \see VNCDiscoverySDKAddDiscovererListener
 */
typedef VNCDiscoverySDKRequestId VNCCALL
VNCDiscoverySDKPostEntityValueRespondToListener(
                              vnc_uint32_t listenerId,
                              VNCDiscoverySDKDiscoverer *pDiscoverer,
                              VNCDiscoverySDKEntity *pEntity,
                              const char *pKey,
                              const char *pValue,
                              VNCDiscoverySDKTimeoutMicroseconds timeOut);

/**
 * \brief Retrieves the string representing the type of the Discoverer.
 *
 * This method reurns a string that represents the Discoverer type. This is the
 * type given when calling VNCDiscoverySDKCreateDiscoverer.
 *
 * \param pDiscoverer The Discoverer for which to retrieve the type. MUST not
 * be NULL (otherwise a crash will occur).
 *
 * \return The string containing the type of the Discoverer. This string is
 * created for the caller, and ownership is passed to it. It is the caller's
 * responsibility to free the string using VNCDiscoverySDKFreeString when
 * finished with it. If there is not enough memory to allocate the string,
 * a NULL pointer will be returned.
 *
 * \see VNCDiscoverySDKCreateDiscoverer
 */
typedef char* VNCCALL
VNCDiscoverySDKGetDiscovererType(VNCDiscoverySDKDiscoverer *pDiscoverer);

/**
 * \brief Registers a Discoverer that is statically linked into the application.
 *
 * The preferred way to implement a Discoverer is to link it into a DLL or shared
 * object that exports the VNCDiscovererInitialize() entry point. This allows reuse
 * of the Discoverer from any Viewer SDK application, without recompilation.
 *
 * VNCDiscoverySDKRegisterDiscoverer() is provided for use when that approach is not
 * possible (e.g. if the platform on which the Discovery SDK is running does not
 * support runtime dynamic linking).
 *
 * \param pSDKInstance The instance of the SDK for which the Discoverer will be 
 *					   registered.
 * \param pType The string decribing the type of Discoverer to register. This is
 *              owned by the client application and the type must be unique.
 * \param pDiscovererInitializeType The address of the Discoverer initialization function.
 *
 * \return VNCDiscoverySDKErrorNone if the Discoverer is registered successfully,
 *         VNCDiscoverSDKErrorOutOfMemory if there is not enough memory to
 *         register the new Discoverer, VNCDiscoverySDKErrorInvalidParameter if the
 *         SDK instance passed in, or pType, is NULL.
 *
 * \see VNCDiscoverySDKInitialize
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKRegisterDiscoverer(VNCDiscoverySDKInstance *pSDKInstance,
                      const char *pType,
                      VNCDiscovererInitializeType *pDiscovererInitializeType);

/**
 * \brief Offers a device to a particular discoverer.
 *
 * This call should be made in response to
 * VNCDiscoverySDKChooseDiscovererForDevice.
 *
 * \param pSDKInstance The instance of the SDK that requested the client
 * application decision.
 * \param pDevice The device that was offered to the client application.
 * \param pDiscoverer The Discoverer that should receive the device. Can be
 * NULL if the application decides to not offer the device to any discoverer.
 *
 * \see VNCDiscoverySDKOfferDeviceToDiscovererForProtocol,
 * VNCDiscoverySDKChooseDiscovererForDevice, VNCDiscoverySDKDevice
 */
typedef void VNCCALL
VNCDiscoverySDKOfferDeviceToDiscoverer(VNCDiscoverySDKInstance *pSDKInstance,
                      const VNCDiscoverySDKDevice *pDevice,
                      VNCDiscoverySDKDiscoverer *pDiscoverer);

/**
 * \brief Offers a device to a particular discoverer specifying the protocol to
 * be used.
 *
 * This call is an extension of the functionality provided by
 * VNCDiscoverySDKOfferDeviceToDiscoverer. Since discoverers can be used for
 * different protocols, this method provides a way of explicitly stating which
 * protocol should be used by the discoverer when connecting to the device.
 *
 * Ownership of pParameters remains with the caller of the method, and the
 * object pointed to can disappear at any point after this method returns.
 *
 * This call should be made in response to
 * VNCDiscoverySDKChooseDiscovererForDevice.
 *
 * \param pSDKInstance The instance of the SDK that requested the client
 * application decision.
 * \param pDevice The device that was offered to the client application.
 * \param pDiscoverer The Discoverer that should receive the device. Can be
 * NULL if the application decides to not offer the device to any discoverer.
 * \param pParameters The parameters that specify the chosen protocol.
 *
 * \see VNCDiscoverySDKChooseDiscovererForDevice, VNCDiscoverySDKDevice
 */
typedef void VNCCALL
VNCDiscoverySDKOfferDeviceToDiscovererForProtocol(
                      VNCDiscoverySDKInstance *pSDKInstance,
                      const VNCDiscoverySDKDevice *pDevice,
                      VNCDiscoverySDKDiscoverer *pDiscoverer,
                      const VNCDiscovererOfferParameter* pParameters);

/**
 * \brief Notifies the Discovery SDK that it can now free the resources
 * associated with the specified device.
 *
 * This is an asynchronous call.
 *
 * If the 'AcknowledgeCancelledDeviceChoice' property is set and this call is
 * not made in response to VNCDiscoverySDKCancelDeviceChoice the device will
 * live in memory until the DiscoverySDK is destroyed.
 *
 * \param pDevice The device that should be destroyed.
 * \see VNCDiscoverySDKCancelDeviceChoice
 */
typedef void VNCCALL
VNCDiscoverySDKAcknowledgeCancelledDeviceChoice(
                      VNCDiscoverySDKInstance *pSDKInstance,
                      const VNCDiscoverySDKDevice *pDevice);

/**
 * \brief Adds a VNC license to a Discovery SDK instance.
 *
 * The Discovery SDK must be licensed in order to enable the user of certain
 * features and to allow the use of certain types of discoverers. To obtain a
 * VNC license, contact your RealVNC sales representative.
 *
 * Calls to VNCDiscoverySDKAddLicense() should be made before creating any
 * discoverers.
 *
 * \param pSDKInstance The instance of the SDK to add the license to.
 *
 * \param licenseText The text of the license. This is a NULL-terminated
 *                    string, and should contain the entire license including
 *                    the header and footer. If your license has been supplied
 *                    to you in a .vnclicense file, then licenseText should
 *                    contain the entire contents of that file.
 *
 * \param serialNumber If serialNumber is not NULL, then, on successful return
 *                     it contains the license serial number, which is a
 *                     big-endian UUID. If an error occurs, this will be
 *                     0-filled. If the serial number is not required, then set
 *                     serialNumber to NULL.
 *
 * \param serialNumberSize The size in bytes of the buffer pointed to by
 *                         serialNumber. If serialNumber is not NULL, then
 *                         serialNumberSize should be at least 16. If
 *                         serialNumber is NULL, then serialNumberSize should
 *                         be 0. If the size is greater than the UUID size,
 *                         then the remainder of the buffer will be 0-filled.
 *
 * \return VNCDiscoverySDKErrorNone if the license is added successfully,
 *          VNCDiscoverySDKErrorInvalidParameter if pSDKInstance or
 *          licenseText are NULL, or the combination of serialNumber and
 *          serialNumberSize parameters is not valid, or
 *          VNCDiscoverySDKErrorLicenseNotValid if licenseText does not contain
 *          a valid VNC license.
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKAddLicense(VNCDiscoverySDKInstance *pSDKInstance,
                      const char *licenseText,
                      vnc_uint8_t *serialNumber,
                      size_t serialNumberSize);

/**
 * Returns a string explanation of the error code.
 *
 * \param error The error code to convert to string.
 *
 * \return The string representation of the error code. If the error code is not known this
 * information will be contained in the returned string.
 */
typedef const char* VNCCALL
VNCDiscoverySDKErrorToString(VNCDiscoverySDKError error);

/**
 * \brief Queries the value of a VNC Discovery SDK parameter.
 *
 * The configuration of a VNC Discovery SDK is stored internally as a list of
 * name-value pairs.  This method queries the value of a VNC Discovery SDK parameter
 * and returns it.  The parameter names are case-insensitive.
 *
 * \param pSDKInstance The VNC Discovery SDK instance whose parameter to set.
 * \param parameter The name of the VNC Viewer parameter to set as a UTF-8
 * encoded, NUL-terminated string.  See vncdiscoverysdkparameters.h for an
 * explanation of the supported parameters.
 *
 * \return Returns a copy of the parameter's value as a UTF-8 encoded,
 * NUL-terminated string.  When you are finished with the copied value, you
 * must free it by calling VNCDiscoverySDKFreeString().  Returns NULL if the
 * parameter name is not valid or if memory could not be allocated.
 *
 * \note The values of parameters whose values are integers are formatted into
 * strings of decimal digits.
 *
 * \see VNCDiscoverySDKSetParameter, vncdiscoverysdkparameters.h
 */
typedef char* VNCCALL
VNCDiscoverySDKGetParameter(VNCDiscoverySDKInstance *pSDKInstance,
    const char* parameter);

/**
 * \brief Sets the value of a VNC Discovery SDK parameter.
 *
 * The parameters must be set before loading any Discoverers.
 *
 * VNC Discovery SDK parameters may never have NULL values.  To 'clear' the
 * value of a string parameter, pass an empty string as the value.
 *
 * Parameters whose values are integers are formatted in decimal.
 *
 * VNC Discovery SDK parameter names are case-insensitive.
 *
 * \param pSDKInstance The VNC Discovery SDK instance whose parameter to set.
 * \param parameter The name of the VNC Discovery SDK parameter to set as a UTF-8
 * encoded, NUL-terminated string.  See vncdiscoverysdkparameters.h for an
 * explanation of the supported parameters.
 * \param value The new parameter value as a UTF-8 encoded, NUL-terminated
 * string.  You may free this string after VNCDiscoverySDKSetParameter() has
 * returned.
 *
 * \retval VNCDiscoverySDKErrorNone The parameter was set successfully.
 * \retval VNCDiscoverySDKErrorInvalidParameter Either the parameter name is not
 * recognized, or the given value is not valid for this parameter.
 * \retval VNCDiscoverySDKErrorIllegalWhileRunning The parameter was being set
 * when the SDK is running (at least one Discoverer has been created).
 *
 * \see VNCDiscoverySDKGetParameter, vncdiscoverysdkparameters.h
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKSetParameter(VNCDiscoverySDKInstance *pSDKInstance,
    const char* parameter,
    const char* value);

/**
 * \brief Retrieves the utility functions supported by the SDK.
 *
 * \param pSDKInstance The SDK instance used to retrieve the utility functions.
 * \param pUtils A pointer to a structure which will be filled with the utility
 * functions.
 * \param utilsSize The size of the structure.
 *
 * \see vncdiscoverysdkutils.h
 * \see VNCDiscoverySDKUtils
 */
typedef void VNCCALL
VNCDiscoverySDKGetUtils(VNCDiscoverySDKInstance *pSDKInstance,
    VNCDiscoverySDKUtils* pUtils, size_t utilsSize);

/**
 * \brief Registers a Discoverer with a dynamic context.
 *
 * The preferred way to implement a Discoverer is to link it into a DLL or shared
 * object that exports the VNCDiscovererInitialize() entry point. This allows reuse
 * of the Discoverer from any Viewer SDK application, without recompilation.
 *
 * VNCDiscoverySDKRegisterDiscovererEx() is provided for use when that approach is not
 * possible (e.g. if the platform on which the Discovery SDK is running does not
 * support runtime dynamic linking).
 * 
 * This method has an additional parameter for the discoverer's dynamic context,
 * which is a pointer passed through the SDK to the discoverer (the discoverer
 * accesses this by calling a support API). Hence this API provides a means
 * to share runtime state with the initialised discoverer.
 *
 * \param pSDKInstance The instance of the SDK for which the Discoverer will be 
 *                                         registered.
 * \param pType The string decribing the type of Discoverer to register. This is
 *              owned by the client application and the type must be unique.
 * \param pDiscovererInitializeType The address of the Discoverer initialization function.
 * \param pDiscovererDynamicContext The dynamic context of the Discoverer.
 *
 * \return VNCDiscoverySDKErrorNone if the Discoverer is registered successfully,
 *         VNCDiscoverSDKErrorOutOfMemory if there is not enough memory to
 *         register the new Discoverer, VNCDiscoverySDKErrorInvalidParameter if the
 *         SDK instance passed in, or pType, is NULL.
 *
 * \see VNCDiscoverySDKInitialize
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKRegisterDiscovererEx(VNCDiscoverySDKInstance *pSDKInstance,
                      const char *pType,
                      VNCDiscovererInitializeType *pDiscovererInitializeType,
                      VNCDiscovererDynamicContext pDiscovererDynamicContext);

/**
 * \brief The interface available to client application for discovering VNC
 * Servers, or doing any discovery-related opearations.
 */
typedef struct
{
  /** Major component of the SDK's version number. */
  vnc_int32_t versionMajor;
  /** Minor component of the SDK's version number. */
  vnc_int32_t versionMinor;
  /** Patch component of the SDK's version number. */
  vnc_int32_t versionPatch;
  /** Build number component of the SDK's version number. */
  vnc_int32_t versionBuild;

  /**
   * Associates a context pointer with a Discovery SDK instance.
   */
  VNCDiscoverySDKSetContext *setContext;
  /**
   * Retrieves the Discovery SDK instance's context pointer.
   */
  VNCDiscoverySDKGetContext *getContext;
  /**
   * Sets the level of logging that will be output by the SDK.
   */
  VNCDiscoverySDKSetLoggingSeverity *setLoggingSeverity;
  /* Discoverer creating and configuring */
  /**
   * Creates a new Discoverer instance of a specific type.
   */
  VNCDiscoverySDKCreateDiscoverer *createDiscoverer;
  /**
   * Associates a context pointer with a Discoverer instance.
   */
  VNCDiscoverySDKDiscovererSetContext *discovererSetContext;
  /**
   * Retrieves the Discoverer instance's context pointer.
   */
  VNCDiscoverySDKDiscovererGetContext *discovererGetContext;
  /**
   * Sets a specific property on a Discoverer instance.
   */
  VNCDiscoverySDKDiscovererSetProperty *discovererSetProperty;
  /**
   * Gets a specific property from a Discoverer instance.
   */
  VNCDiscoverySDKDiscovererGetProperty *discovererGetProperty;
  /* Discoverer discovery calls */
  /**
   * Tells a Discoverer to start discovering.
   */
  VNCDiscoverySDKDiscovererStart *discovererStart;
  /**
   * Tells a Discoverer to stop discovering.
   */
  VNCDiscoverySDKDiscovererStop *discovererStop;
  /**
   * Retrieves the Viewer Parameters needed in addition to vnccmd.
   */
  VNCDiscoverySDKGetViewerParameters *getViewerParameters;
  /* Working with entities (values and context data) */
  /**
   * Fetches the value for a key of an entity.
   */
  VNCDiscoverySDKFetchEntityValue *fetchEntityValue;
  /**
   * Gets the value for a key of an entity synchronously.
   */
  VNCDiscoverySDKFetchEntityValueBlocking *fetchEntityValueBlocking;
  /**
   * Gets some binary data owned by an entity.
   */
  VNCDiscoverySDKGetEntityBinaryData *getEntityBinaryData;
  /**
   * Posts a value to a key of an entity.
   */
  VNCDiscoverySDKPostEntityValue *postEntityValue;
  /**
   * Posts a value synchronously to a key of an entity.
   */
  VNCDiscoverySDKPostEntityValueBlocking *postEntityValueBlocking;
  /**
   * Associates a context pointer with an Entity.
   */
  VNCDiscoverySDKEntitySetContext *entitySetContext;
  /**
   * Retrieve the Entity's context pointer.
   */
  VNCDiscoverySDKEntityGetContext *entityGetContext;
  /* Calls needed for Adressbook Discoverer */
  /**
   * Creates an entity to be published by an address book Discoverer.
   */
  VNCDiscoverySDKCreateEntity *createEntity;
  /**
   * Publishes an entity by an address book Discoverer.
   */
  VNCDiscoverySDKPublishEntity *publishEntity;
  /**
   * Unpublishes (removes) an entity by an address book Discoverer.
   */
  VNCDiscoverySDKUnpublishEntity *unpublishEntity;
  /**
   * Destroys an entity previously created to be published by an address-book 
   * Discoverer.
   */
  VNCDiscoverySDKDestroyUnpublishedEntity *destroyUnpublishedEntity;
  /* Cleaning up */
  /**
   * Destroys a Discoverer.
   */
  VNCDiscoverySDKDiscovererDestroy *discovererDestroy;
  /**
   * Destroys the SDK.
   */
  VNCDiscoverySDKDestroy *destroy;
  /* Support for allocating and freeing */
  /**
   * Allocates the memory for a string.
   */
  VNCDiscoverySDKAllocString *allocString;
  /**
   * Frees the memory of a string allocated by the SDK.
   */
  VNCDiscoverySDKFreeString *freeString;
  /**
   * Frees the memory of a key-value array allocated by the SDK.
   */
  VNCDiscoverySDKFreeKeyValPairArray *freeArray;
  /**
   * Returns the current status of the Discoverer.
   */
  VNCDiscoverySDKGetDiscovererStatus *getDiscovererStatus;
  /**
   * Adds a new set of callbacks for the a Discoverer.
   */
  VNCDiscoverySDKAddDiscovererListener *addDiscovererListener;
  /**
   * Removes a listener from a Discoverer.
   */
  VNCDiscoverySDKRemoveDiscovererListener *removeDiscovererListener;
  /**
   * Fetches the value for a key of an entity and sends the response to a
   * particular listener.
   */
  VNCDiscoverySDKFetchEntityValueRespondToListener *fetchEntityValueRespondToListener;
  /**
   * Posts a value to a key of an entity and reports the outcome to a
   * particular listener.
   */
  VNCDiscoverySDKPostEntityValueRespondToListener *postEntityValueRespondToListener;
  /**
   * Retrieves the string representing the type of the Discoverer.
   */
  VNCDiscoverySDKGetDiscovererType *getDiscovererType;
  /**
   * Registers a new Discoverer of a specific type.
   */
  VNCDiscoverySDKRegisterDiscoverer *registerDiscoverer;
  /**
   * Offers a device to a particular discoverer.
   */
  VNCDiscoverySDKOfferDeviceToDiscoverer *offerDeviceToDiscoverer;
  /**
   * Returns a string explanation of the error code.
   */
  VNCDiscoverySDKErrorToString *errorToString;
  /**
   * Adds a VNC license to the Discovery SDK instance.
   */
  VNCDiscoverySDKAddLicense *addLicense;
  /**
   * Queries the value of a VNC Discovery SDK parameter.
   */
  VNCDiscoverySDKGetParameter *getParameter;
  /**
   * Sets the value of a VNC Discovery SDK parameter.
   */
  VNCDiscoverySDKSetParameter *setParameter;
  /**
   * Retrieves the utility functions supported by the SDK.
   */
  VNCDiscoverySDKGetUtils *getUtils;
  /**
   * Registers a new Discoverer with a dynamic context.
   */
  VNCDiscoverySDKRegisterDiscovererEx *registerDiscovererEx;
  /**
   * Tells multiple discoverers to start discovering.
   */
  VNCDiscoverySDKDiscovererStartMultiple *discovererStartMultiple;
  /**
   * Tells multiple discoverers to stop discovering.
   */
  VNCDiscoverySDKDiscovererStopMultiple *discovererStopMultiple;
  /**
   * Notifies the Discovery SDK that it can now free the
   * resources associated with the specified entity
   */
  VNCDiscoverySDKAcknowledgeEntityDisappeared *acknowledgeEntityDisappeared;
  /**
   * Tells a Discoverer to start discovering and return error if
   * the request cannot be handled.
   */
  VNCDiscoverySDKDiscovererStartEx *discovererStartEx;
  /**
   * Tells a Discoverer to stop discovering and return error if
   * the request cannot be handled.
   */
  VNCDiscoverySDKDiscovererStopEx *discovererStopEx;
  /**
   * Tells multiple discoverers to start discovering and return error
   * if it cannot be handled.
   */
  VNCDiscoverySDKDiscovererStartMultipleEx *discovererStartMultipleEx;
  /**
   * Tells multiple discoverers to stop discovering and return error
   * if it cannot be handled.
   */
  VNCDiscoverySDKDiscovererStopMultipleEx *discovererStopMultipleEx;
  /**
   * Offers a device to a particular discoverer specifying a protocol.
   */
  VNCDiscoverySDKOfferDeviceToDiscovererForProtocol *offerDeviceForProtocol;
  /**
   * Notifies the Discovery SDK that it can now free the resources associated
   * with the specified device
   */
  VNCDiscoverySDKAcknowledgeCancelledDeviceChoice *acknowledgeCancelledDeviceChoice;
} VNCDiscoverySDK;

/**
 * \brief The type of the sole entry point into the Discovery SDK.
 *
 * This is the type of the sole entry point into the Discovery SDK DLL or
 * shared object.
 *
 * \param pSDK A pointer to the structure that gets populated by the SDK to
 *             enable the client application to make discovery requests.
 * \param sdkSize The size of the SDK structure passed in.
 * \param pCallbacks A pointer to the structure that holds the callback
 *                   pointers which the SDK will use when responding to
 *                   asynchronous requests, or making any notifications.
 * \param callbacksSize The size of the callbacks structure.
 *
 * \return A pointer to the SDK Instance. IF this pointer is NULL then the SDK
 *         Instance can't be created because there is not enough memory.
 *
 * \see VNCDiscoverySDKCallbacks
 */
typedef VNCDiscoverySDKInstance *VNCCALL
VNCDiscoverySDKInitializeType(VNCDiscoverySDK *pSDK, 
                      size_t sdkSize,
                      VNCDiscoverySDKCallbacks *pCallbacks,
                      size_t callbacksSize);

/**
 * \brief Declaration of the sole entry point.
 *
 * This is the sole entry point to the Discovery SDK. The client application
 * needs to call this to be able to use the Discovery interface.
 *
 * \param pSDK A pointer to the structure that gets populated by the SDK to
 *             enable the client application to make discovery requests.
 * \param sdkSize The size of the SDK structure passed in.
 * \param pCallbacks A pointer to the structure that holds the callback
 *                   pointers which the SDK will use when responding to
 *                   asynchronous requests, or making any notifications.
 * \param callbacksSize The size of the callbacks structure.
 *
 * \return A pointer to the SDK Instance. IF this pointer is NULL then the SDK
 *         Instance can't be created because there is not enough memory.
 *
 * \see VNCDiscoverySDKInitializeType
 */
VNCDLLIMPORT VNCDiscoverySDKInstance *VNCCALL
VNCDiscoverySDKInitialize(VNCDiscoverySDK *pSDK, 
                      size_t sdkSize,
                      VNCDiscoverySDKCallbacks *pCallbacks,
                      size_t callbacksSize);

#ifdef __cplusplus
}
#endif



#endif /* !defined(__VNCDISCOVERYSDK_H__) */

